<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens; // <-- ESTA É A LINHA QUE FALTAVA
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

// Adiciona todos os models que o User precisa de "conhecer"
use App\Models\Carteira;
use App\Models\Transacao;
use App\Models\InvestimentoUsuario;
use App\Models\TicketSuporte;
use App\Models\ComissaoIndicacao;
use App\Models\PerfilKyc;

class User extends Authenticatable
{
    // Esta linha usa o Trait que importámos em cima
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * O nome da tabela associada ao model.
     */
    protected $table = 'usuarios';

    /**
     * Os atributos que podem ser preenchidos em massa.
     */
    protected $fillable = [
        'nome_completo',
        'email',
        'senha',
        'tipo_usuario',
        'status',
        'telefone',
        'indicado_por_usuario_id',
        'codigo_indicacao',
        'email_verificado_em',
    ];

    /**
     * Os atributos que devem ser escondidos.
     */
    protected $hidden = [
        'senha',
        'remember_token',
        'token_2fa_secreto',
    ];

    /**
     * Os atributos que devem ser convertidos.
     */
    protected $casts = [
        'email_verificado_em' => 'datetime',
        'senha' => 'hashed',
        'usa_2fa' => 'boolean',
    ];

    /**
     * Mutator: Criptografa a senha.
     */
    public function setSenhaAttribute($value)
    {
        $this->attributes['senha'] = Hash::make($value);
    }

    /**
     * Boot: Gera o código de indicação.
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($model) {
            if (empty($model->codigo_indicacao)) {
                $model->codigo_indicacao = 'CLUB-' . Str::upper(Str::random(8));
            }
        });
    }

    // --- RELAÇÕES ---

    public function carteira()
    {
        return $this->hasOne(Carteira::class, 'usuario_id');
    }

    public function transacoes()
    {
        return $this->hasMany(Transacao::class, 'usuario_id');
    }

    public function investimentos()
    {
        return $this->hasMany(InvestimentoUsuario::class, 'usuario_id');
    }

    public function tickets()
    {
        return $this->hasMany(TicketSuporte::class, 'usuario_id');
    }

    /**
     * Define a relação: Um Utilizador TEM MUITOS Utilizadores que ele indicou (os "filhos")
     */
    public function indicados()
    {
        return $this->hasMany(User::class, 'indicado_por_usuario_id');
    }

    /**
     * Define a relation: Um Utilizador TEM MUITAS Comissões (que ele recebeu)
     */
    public function comissoes()
    {
        return $this->hasMany(ComissaoIndicacao::class, 'usuario_id');
    }

    /**
     * Define a relação: Um Utilizador TEM UM Perfil KYC
     */
    public function kyc()
    {
        return $this->hasOne(PerfilKyc::class, 'usuario_id');
    }
}