<?php

namespace App\Http\Controllers\Api\Cliente;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\TicketSuporte;
use App\Models\MensagemTicket;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str; // Para gerar o ID único do ticket

class SuporteController extends Controller
{
    /**
     * Função LISTAR TICKETS (GET /api/suporte/tickets)
     * Lista todos os tickets abertos pelo utilizador logado.
     */
    public function listarTickets(Request $request)
    {
        $user = Auth::user();
        
        $tickets = $user->tickets()
                        ->orderBy('ultima_resposta_em', 'desc')
                        ->paginate(15);

        return response()->json($tickets, 200);
    }

    /**
     * Função ABRIR TICKET (POST /api/suporte/tickets/abrir)
     * Cria um novo ticket de suporte e a sua primeira mensagem.
     */
    public function abrirTicket(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'assunto' => 'required|string|max:255',
            'mensagem' => 'required|string',
            'prioridade' => 'nullable|in:baixa,media,alta', // Valida a prioridade
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $user = Auth::user();

        // Usamos DB::transaction para garantir que criamos o Ticket E a Mensagem
        try {
            DB::beginTransaction();

            // 1. Cria o Ticket principal
            $ticket = $user->tickets()->create([
                'ticket_uid' => 'TKT-' . Str::upper(Str::random(8)), // Gera um ID único
                'assunto' => $request->assunto,
                'status' => 'aberto', // Status inicial
                'prioridade' => $request->prioridade ?? 'media',
                'ultima_resposta_em' => now(), // Agora
            ]);

            // 2. Cria a primeira mensagem
            $ticket->mensagens()->create([
                'usuario_id' => $user->id, // A mensagem é do utilizador
                'mensagem' => $request->mensagem,
                // 'anexo_path' => null // Lógica de upload de ficheiro viria aqui
            ]);

            DB::commit(); // Confirma as criações

            return response()->json([
                'message' => 'Ticket aberto com sucesso!',
                'ticket' => $ticket
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack(); // Reverte em caso de erro
            return response()->json(['message' => 'Erro ao abrir o ticket.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Função VER TICKET (GET /api/suporte/tickets/{id})
     * Vê um ticket específico e todas as suas mensagens.
     */
    public function verTicket(Request $request, $id)
    {
        $user = Auth::user();

        // 1. Busca o ticket, mas garante que pertence ao utilizador logado
        $ticket = $user->tickets()
                       ->with([
                           // 2. Carrega as mensagens e o autor de cada mensagem
                           'mensagens.user:id,nome_completo' 
                       ])
                       ->find($id); // Procura pelo ID

        if (!$ticket) {
            return response()->json(['message' => 'Ticket não encontrado.'], 404);
        }

        return response()->json($ticket, 200);
    }

    /**
     * Função RESPONDER TICKET (POST /api/suporte/tickets/{id}/responder)
     * Adiciona uma nova mensagem a um ticket existente.
     */
    public function responderTicket(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'mensagem' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $user = Auth::user();

        // 1. Busca o ticket, garantindo que pertence ao utilizador
        $ticket = $user->tickets()->find($id);

        if (!$ticket) {
            return response()->json(['message' => 'Ticket não encontrado.'], 404);
        }

        // 2. Verifica se o ticket não está fechado
        if ($ticket->status === 'fechado') {
            return response()->json(['message' => 'Este ticket está fechado e não pode ser respondido.'], 403);
        }

        // 3. Cria a nova mensagem
        $mensagem = $ticket->mensagens()->create([
            'usuario_id' => $user->id,
            'mensagem' => $request->mensagem,
            // 'anexo_path' => null // Lógica de upload de ficheiro viria aqui
        ]);

        // 4. Atualiza o status do ticket
        // Quando o cliente responde, o status volta para "aberto" (aguardando admin)
        $ticket->status = 'aberto';
        $ticket->ultima_resposta_em = now();
        $ticket->save();

        return response()->json([
            'message' => 'Resposta enviada!',
            'mensagem' => $mensagem
        ], 201);
    }
}