<?php

namespace App\Http\Controllers\Api\Cliente;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Oportunidade;
use Illuminate\Support\Facades\Auth;       
use Illuminate\Support\Facades\DB;        
use Illuminate\Support\Facades\Validator;   
use App\Models\InvestimentoUsuario;         

class OportunidadeController extends Controller
{
    /**
     * Função LISTAR (GET /api/oportunidades)
     * Mostra todas as oportunidades ativas (Casos) para o usuário investir.
     */
    public function listar(Request $request)
    {
        // O Eager Loading (com 'with') é crucial para a performance!
        // Ele busca a oportunidade E os seus detalhes numa só consulta.
        $oportunidades = Oportunidade::where('status', 'ativo')
            ->with([
                'metaInvestimento', // Carrega os detalhes de investimento
                'metaImovel',       // Carrega os detalhes de imóvel
                'criador:id,nome_completo' // Carrega só o ID e Nome do criador
            ])
            ->orderBy('created_at', 'desc') // Mais recentes primeiro
            ->paginate(12); // Paginação (12 por página)

        return response()->json($oportunidades, 200);
    }

    /**
     * Função DETALHES (GET /api/oportunidades/{id})
     * Mostra os detalhes completos de UM caso, incluindo as abas de conteúdo.
     */
    public function detalhes(Request $request, $id)
    {
        // Tenta encontrar a oportunidade ativa com o $id
        // O 'with' aqui carrega TUDO (meta, criador E as abas de conteúdo)
        $oportunidade = Oportunidade::where('status', 'ativo')
            ->with([
                'metaInvestimento',
                'metaImovel',
                'criador:id,nome_completo',
                'conteudos' // Carrega as abas (Visão Geral, FAQ, etc.)
            ])
            ->find($id); // Usamos find() em vez de findOrFail()

        // Se não encontrar (ou não estiver 'ativo')
        if (!$oportunidade) {
            return response()->json(['message' => 'Oportunidade não encontrada ou não está ativa.'], 404);
        }

        return response()->json($oportunidade, 200);
    }
// ... (depois da função detalhes())

/**
 * Função INVESTIR (POST /api/oportunidades/investir)
 * Ação principal de investimento do utilizador.
 */
public function investir(Request $request)
{
    $validator = Validator::make($request->all(), [
        'oportunidade_id' => 'required|integer|exists:oportunidades,id',
        'valor' => 'required|numeric|min:1',
    ]);

    if ($validator->fails()) {
        return response()->json($validator->errors(), 422);
    }

    $user = Auth::user();
    $oportunidadeId = $request->oportunidade_id;
    $valorInvestido = (float) $request->valor;

    // Inicia a transação segura do banco de dados
    try {
        DB::beginTransaction();

        // 1. Pega os "atores" da transação e "tranca" as linhas
        // Isto evita que o utilizador faça duas ações ao mesmo tempo
        $carteira = $user->carteira()->lockForUpdate()->first();
        $oportunidade = Oportunidade::with('metaInvestimento')
                            ->where('id', $oportunidadeId)
                            ->lockForUpdate()
                            ->first();

        // 2. --- VERIFICAÇÕES DE NEGÓCIO ---

        // A) A oportunidade existe e está ativa?
        if (!$oportunidade || $oportunidade->status !== 'ativo') {
            DB::rollBack();
            return response()->json(['message' => 'Esta oportunidade não está disponível para investimento.'], 404);
        }

        // B) O utilizador tem saldo suficiente?
        if ($carteira->saldo_disponivel < $valorInvestido) {
            DB::rollBack();
            return response()->json(['message' => 'Saldo insuficiente.'], 422);
        }

        // C) O valor está dentro dos limites Mín/Máx do investimento?
        $meta = $oportunidade->metaInvestimento;
        if ($meta) {
            if ($valorInvestido < $meta->investimento_min) {
                DB::rollBack();
                return response()->json(['message' => 'O valor é inferior ao investimento mínimo.'], 422);
            }
            if ($meta->investimento_max > 0 && $valorInvestido > $meta->investimento_max) {
                DB::rollBack();
                return response()->json(['message' => 'O valor é superior ao investimento máximo.'], 422);
            }
        }

        // D) O utilizador já investiu? (Baseado na nossa migration 'unique')
        $jaInvestiu = InvestimentoUsuario::where('usuario_id', $user->id)
                                         ->where('oportunidade_id', $oportunidadeId)
                                         ->exists();
        if ($jaInvestiu) {
            DB::rollBack();
            return response()->json(['message' => 'Você já investiu nesta oportunidade.'], 409); // 409 = Conflito
        }


        // 3. --- EXECUÇÃO FINANCEIRA ---
        // Todas as verificações passaram. Vamos mover o dinheiro.

        // A) Move o dinheiro na carteira do utilizador
        $carteira->saldo_disponivel -= $valorInvestido;
        $carteira->saldo_bloqueado += $valorInvestido;
        $carteira->save();

        // B) Atualiza o valor arrecadado na oportunidade
        $oportunidade->valor_arrecadado += $valorInvestido;

        // (Lógica Bónus) Se a meta foi atingida, fecha a oportunidade
        if ($oportunidade->valor_arrecadado >= $oportunidade->valor_meta) {
            $oportunidade->status = 'financiado';
        }
        $oportunidade->save();

        // 4. --- CRIAÇÃO DOS REGISTOS ---

        // A) Cria o registo da Transação (para o histórico)
        $transacao = $user->transacoes()->create([
            'tipo_transacao' => 'investimento',
            'status' => 'concluido',
            'valor' => $valorInvestido,
        ]);

        // B) Cria o registo do Investimento (para "Meus Investimentos")
        $investimento = $user->investimentos()->create([
            'oportunidade_id' => $oportunidadeId,
            'transacao_id' => $transacao->id,
            'valor_investido' => $valorInvestido,
            'status' => 'ativo',
            // Define a data de conclusão (se existir no meta)
            'data_conclusao_prevista' => $meta->duracao_investimento ? now()->addMonths(intval($meta->duracao_investimento)) : null,
        ]);

        // 5. SUCESSO! Confirma tudo no banco de dados
        DB::commit();

        return response()->json([
            'message' => 'Investimento realizado com sucesso!',
            'investimento' => $investimento->load('oportunidade'),
            'nova_carteira' => $carteira,
        ], 201);

    } catch (\Exception $e) {
        // 6. FALHA. Reverte tudo o que foi feito
        DB::rollBack();
        return response()->json(['message' => 'Ocorreu um erro ao processar seu investimento.', 'error' => $e->getMessage()], 500);
    }
}




   
}