<?php

namespace App\Http\Controllers\Api\Cliente;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth; // Para pegar o utilizador logado
use Illuminate\Support\Facades\DB;   // Para transações de BD seguras
use Illuminate\Support\Facades\Validator;

class FinanceiroController extends Controller
{
    /**
     * Função HISTÓRICO (GET /api/minhas-transacoes)
     * Mostra a lista de TODAS as transações (Depósito, Retirada, Lucro, etc.)
     */
    public function historico(Request $request)
    {
        $user = Auth::user();

        $transacoes = $user->transacoes()
                           ->orderBy('created_at', 'desc') // Mais recentes primeiro
                           ->paginate(20); // Paginação

        return response()->json($transacoes, 200);
    }

    /**
     * Função EXTRATO DE LUCRO (GET /api/meu-extrato-lucro)
     * Mostra apenas as transações de lucro (ROI)
     */
    public function extratoLucro(Request $request)
    {
        $user = Auth::user();

        $lucros = $user->transacoes()
                       ->where('tipo_transacao', 'retorno_lucro')
                       ->where('status', 'concluido')
                       ->orderBy('created_at', 'desc')
                       ->paginate(20);

        return response()->json($lucros, 200);
    }

    /**
     * Função INICIAR DEPÓSITO (POST /api/deposito/iniciar)
     * Cria uma solicitação de depósito que o Admin irá aprovar.
     */
    public function iniciarDeposito(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'valor' => 'required|numeric|min:1', // Deve ser um número maior que 0
            'gateway_pagamento' => 'required|string', // Ex: "PIX", "Manual", "PayPal"
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $user = Auth::user();

        // Cria a transação com status 'pendente'
        $transacao = $user->transacoes()->create([
            'tipo_transacao' => 'deposito',
            'status' => 'pendente', // O Admin deve aprovar
            'valor' => $request->valor,
            'gateway_pagamento' => $request->gateway_pagamento,
        ]);

        return response()->json([
            'message' => 'Solicitação de depósito recebida. Aguardando aprovação.',
            'transacao' => $transacao
        ], 201);
    }

    /**
     * Função SOLICITAR RETIRADA (POST /api/retirada/solicitar)
     * Cria uma solicitação de retirada, verificando e bloqueando o saldo.
     */
    public function solicitarRetirada(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'valor' => 'required|numeric|min:1', // Valor mínimo para retirar
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $user = Auth::user();
        $valorRetirada = (float) $request->valor;

        // Inicia uma Transação de Banco de Dados para segurança.
        // Se algo falhar, tudo é revertido (rollback).
        try {
            DB::beginTransaction();

            // 1. Pega a carteira do usuário com 'lockForUpdate'
            // Isto "tranca" a linha da carteira no BD, evitando que o utilizador
            // faça duas retiradas ao mesmo tempo (condição de corrida).
            $carteira = $user->carteira()->lockForUpdate()->first();

            // 2. Verifica se há saldo
            if ($carteira->saldo_disponivel < $valorRetirada) {
                DB::rollBack(); // Reverte a transação
                return response()->json(['message' => 'Saldo insuficiente.'], 422);
            }

            // 3. O saldo é suficiente. Move o dinheiro do 'disponível' para 'bloqueado'.
            // O dinheiro fica "bloqueado" até o Admin aprovar a retirada.
            $carteira->saldo_disponivel -= $valorRetirada;
            $carteira->saldo_bloqueado += $valorRetirada;
            $carteira->save();

            // 4. Cria o registo da transação de retirada
            $transacao = $user->transacoes()->create([
                'tipo_transacao' => 'retirada',
                'status' => 'pendente', // Admin deve aprovar
                'valor' => $valorRetirada,
            ]);

            // 5. Se tudo correu bem, confirma as alterações no BD
            DB::commit();

            return response()->json([
                'message' => 'Solicitação de retirada recebida. Aguardando aprovação.',
                'transacao' => $transacao,
                'nova_carteira' => $carteira // Retorna o novo saldo
            ], 201);

        } catch (\Exception $e) {
            // 6. Se algo deu errado, reverte tudo
            DB::rollBack();
            return response()->json(['message' => 'Erro ao processar a solicitação. Tente novamente.', 'error' => $e->getMessage()], 500);
        }
    }
}