<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Configuracao;
use App\Models\ComissaoIndicacao;
use App\Models\Carteira;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ReferenciaController extends Controller
{
    /**
     * BUSCAR as configurações de referência
     * (GET /api/admin/referencias/config)
     */
    public function getConfig(Request $request)
    {
        // Busca a % de comissão que definimos no FinanceiroController
        $config = Configuracao::find('comissao_indicacao_percentagem');
        $percentagem = $config ? $config->valor : '5.0'; // Padrão 5%

        return response()->json(['percentagem' => (float) $percentagem], 200);
    }

    /**
     * SALVAR as configurações de referência
     * (POST /api/admin/referencias/config)
     */
    public function setConfig(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'percentagem' => 'required|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        // Salva a % na tabela de configurações (chave-valor)
        Configuracao::updateOrCreate(
            ['chave' => 'comissao_indicacao_percentagem'],
            ['valor' => $request->percentagem]
        );

        return response()->json(['message' => 'Configuração salva com sucesso.'], 200);
    }

    /**
     * LISTAR as comissões (pendentes, pagas, etc.)
     * (GET /api/admin/referencias/comissoes?status=pendente)
     */
    public function listarComissoes(Request $request)
    {
        $query = ComissaoIndicacao::with([
                    'indicador:id,nome_completo,email', // Quem recebe o $
                    'indicado:id,nome_completo' // Quem gerou o $
                ]);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $comissoes = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json($comissoes, 200);
    }

    /**
     * PROCESSAR (Pagar/Rejeitar) uma comissão pendente
     * (POST /api/admin/referencias/comissoes/{id}/processar)
     */
    public function processarComissao(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'acao' => 'required|in:aprovar,rejeitar',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $comissao = ComissaoIndicacao::where('id', $id)
                                    ->where('status', 'pendente')
                                    ->firstOrFail();
        
        // --- Inicia a Transação Segura ---
        try {
            DB::beginTransaction();

            if ($request->acao === 'aprovar') {
                // 1. Pega o indicador (quem recebe) e tranca a carteira
                $indicadorCarteira = Carteira::where('usuario_id', $comissao->usuario_id)
                                            ->lockForUpdate()
                                            ->first();
                
                // 2. Adiciona o saldo ao indicador
                $indicadorCarteira->saldo_disponivel += $comissao->valor_comissao;
                $indicadorCarteira->save();

                // 3. Cria a transação (para o histórico do indicador)
                $indicadorCarteira->user->transacoes()->create([
                    'tipo_transacao' => 'comissao',
                    'status' => 'concluido',
                    'valor' => $comissao->valor_comissao,
                ]);

                // 4. Marca a comissão como paga
                $comissao->status = 'pago';
                $comissao->save();

            } else { // Ação é 'rejeitar'
                // 1. Marca a comissão como cancelada
                $comissao->status = 'cancelado';
                $comissao->save();
                // O dinheiro não se move
            }

            DB::commit(); // Confirma a operação

            return response()->json([
                'message' => 'Comissão processada com sucesso!',
                'comissao' => $comissao
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Erro ao processar comissão.', 'error' => $e->getMessage()], 500);
        }
    }
}