<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Oportunidade;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str; // Para criar o 'slug' (URL amigável)

class OportunidadeController extends Controller
{
    /**
     * LISTAR todas as oportunidades para o Admin (incluindo rascunhos)
     * (GET /api/admin/oportunidades)
     */
    public function index(Request $request)
    {
        $oportunidades = Oportunidade::with('criador:id,nome_completo')
                                    ->orderBy('created_at', 'desc')
                                    ->paginate(20);
        return response()->json($oportunidades, 200);
    }

    /**
     * CRIAR uma nova oportunidade (store)
     * (POST /api/admin/oportunidades)
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'titulo' => 'required|string|max:255',
            'tipo_oportunidade' => 'required|in:investimento,imovel',
            'status' => 'required|in:ativo,rascunho',
            'imagem_destaque' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'meta' => 'required|array', // O objeto com os dados 'meta' (ROI, quartos, etc.)
            'conteudos' => 'required|array|min:1', // As abas (Visão Geral, etc.)
            'conteudos.*.tipo_aba' => 'required|string',
            'conteudos.*.titulo_aba' => 'required|string',
            'conteudos.*.conteudo' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        // Inicia a transação segura
        try {
            DB::beginTransaction();

            // 1. Guarda a Imagem de Destaque (se existir)
            $path = null;
            if ($request->hasFile('imagem_destaque')) {
                // Guarda em 'storage/app/public/oportunidades'
                $path = $request->file('imagem_destaque')->store('oportunidades', 'public');
            }

            // 2. Cria a Oportunidade principal
            $oportunidade = Oportunidade::create([
                'criador_id' => Auth::id(),
                'titulo' => $request->titulo,
                'slug' => Str::slug($request->titulo) . '-' . uniqid(), // Cria uma URL única
                'tipo_oportunidade' => $request->tipo_oportunidade,
                'status' => $request->status,
                'imagem_destaque_path' => $path,
                'valor_meta' => $request->valor_meta ?? 0,
                'data_fim' => $request->data_fim ?? null,
            ]);

            // 3. Cria os dados "Meta" (Investimento ou Imóvel)
            $metaData = $request->meta;
            if ($request->tipo_oportunidade === 'investimento') {
                $oportunidade->metaInvestimento()->create($metaData);
            } else {
                $oportunidade->metaImovel()->create($metaData);
            }

            // 4. Cria os "Conteúdos" (Abas)
            // A função createMany() é perfeita para isto
            $oportunidade->conteudos()->createMany($request->conteudos);

            // 5. Sucesso! Confirma tudo no banco de dados.
            DB::commit();

            return response()->json($oportunidade->load(['metaInvestimento', 'metaImovel', 'conteudos']), 201);

        } catch (\Exception $e) {
            // 6. Falha! Reverte tudo
            DB::rollBack();
            // Apaga a imagem que pode ter sido salva
            if (isset($path)) {
                Storage::disk('public')->delete($path);
            }
            return response()->json(['message' => 'Erro ao criar oportunidade.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * BUSCAR 1 oportunidade (para preencher o formulário de edição)
     * (GET /api/admin/oportunidades/{id})
     */
    public function show($id)
    {
        $oportunidade = Oportunidade::with(['metaInvestimento', 'metaImovel', 'conteudos'])
                                    ->find($id);

        if (!$oportunidade) {
            return response()->json(['message' => 'Oportunidade não encontrada.'], 404);
        }
        return response()->json($oportunidade, 200);
    }

    /**
     * EDITAR uma oportunidade existente (update)
     * (POST /api/admin/oportunidades/{id} -> com _method: 'PUT' no form-data)
     */
    public function update(Request $request, $id)
    {
        // Nota: O update com ficheiros é complexo. O frontend deve enviar
        // um POST com um campo '_method' = 'PUT' ou 'PATCH'.
        
        $oportunidade = Oportunidade::find($id);
        if (!$oportunidade) {
            return response()->json(['message' => 'Oportunidade não encontrada.'], 404);
        }

        // Validação (semelhante ao store, mas 'imagem_destaque' é opcional)
        $validator = Validator::make($request->all(), [
            'titulo' => 'required|string|max:255',
            'status' => 'required|in:ativo,rascunho,financiado,expirado',
            'imagem_destaque' => 'nullable|image|mimes:jpg,jpeg,png|max:2048', // Só se for trocar
            'meta' => 'required|array',
            'conteudos' => 'required|array|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        try {
            DB::beginTransaction();

            // 1. Atualiza dados principais
            $oportunidade->fill($request->only(['titulo', 'status', 'valor_meta', 'data_fim']));
            $oportunidade->slug = Str::slug($request->titulo) . '-' . $oportunidade->id; // Atualiza o slug
            
            // 2. Verifica se uma NOVA imagem foi enviada
            if ($request->hasFile('imagem_destaque')) {
                // Apaga a imagem antiga
                if ($oportunidade->imagem_destaque_path) {
                    Storage::disk('public')->delete($oportunidade->imagem_destaque_path);
                }
                // Salva a nova imagem
                $oportunidade->imagem_destaque_path = $request->file('imagem_destaque')->store('oportunidades', 'public');
            }
            
            $oportunidade->save();

            // 3. Atualiza os dados "Meta"
            // updateOrCreate é perfeito: atualiza se existir, ou cria se não existir.
            $metaData = $request->meta;
            if ($oportunidade->tipo_oportunidade === 'investimento') {
                $oportunidade->metaInvestimento()->updateOrCreate(['oportunidade_id' => $id], $metaData);
            } else {
                $oportunidade->metaImovel()->updateOrCreate(['oportunidade_id' => $id], $metaData);
            }

            // 4. Atualiza os "Conteúdos" (Abas)
            // A forma mais fácil e segura é apagar os antigos e recriar.
            $oportunidade->conteudos()->delete();
            $oportunidade->conteudos()->createMany($request->conteudos);
            
            DB::commit();
            return response()->json($oportunidade->load(['metaInvestimento', 'metaImovel', 'conteudos']), 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Erro ao atualizar oportunidade.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * APAGAR uma oportunidade
     * (DELETE /api/admin/oportunidades/{id})
     */
    public function destroy($id)
    {
        $oportunidade = Oportunidade::find($id);
        if (!$oportunidade) {
            return response()->json(['message' => 'Oportunidade não encontrada.'], 404);
        }

        try {
            DB::beginTransaction();
            // 1. Apaga a imagem do disco
            if ($oportunidade->imagem_destaque_path) {
                Storage::disk('public')->delete($oportunidade->imagem_destaque_path);
            }
            
            // 2. Apaga o registo do banco
            // As tabelas 'meta' e 'conteudos' serão apagadas automaticamente
            // graças ao 'onDelete('cascade')' que definimos nas migrações.
            $oportunidade->delete();

            DB::commit();
            return response()->json(null, 204); // 204 = No Content (sucesso)

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Erro ao apagar oportunidade.', 'error' => $e->getMessage()], 500);
        }
    }
}