<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Transacao;
use App\Models\Carteira;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Models\Configuracao;      
use App\Models\ComissaoIndicacao; 

class FinanceiroController extends Controller
{
    /**
     * LISTA as transações pendentes para o Admin
     * (GET /api/admin/transacoes/pendentes?tipo=deposito)
     * (GET /api/admin/transacoes/pendentes?tipo=retirada)
     */
    public function listarPendentes(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tipo' => 'required|in:deposito,retirada',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $transacoes = Transacao::where('status', 'pendente')
                                ->where('tipo_transacao', $request->tipo)
                                ->with('user:id,nome_completo,email') // Puxa o nome do user
                                ->orderBy('created_at', 'asc') // Mais antigos primeiro
                                ->paginate(20);

        return response()->json($transacoes, 200);
    }

    /**
     * PROCESSA um depósito (Aprovar ou Rejeitar)
     * (POST /api/admin/transacoes/processar-deposito/{id})
     */
    public function processarDeposito(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'acao' => 'required|in:aprovar,rejeitar',
            'notas_admin' => 'required_if:acao,rejeitar|string|nullable',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $transacao = Transacao::where('id', $id)
                            ->where('tipo_transacao', 'deposito')
                            ->where('status', 'pendente')
                            ->firstOrFail();
        
        $userCarteira = Carteira::where('usuario_id', $transacao->usuario_id)->first();

        // --- Inicia a Transação Segura ---
        try {
            DB::beginTransaction();

            if ($request->acao === 'aprovar') {
                // 1. Atualiza a transação
                $transacao->status = 'concluido';
                $transacao->notas_admin = 'Depósito aprovado.';
                $transacao->save();
                
                // 2. Adiciona o saldo ao utilizador
                $userCarteira->saldo_disponivel += $transacao->valor;
                $userCarteira->save();
                
                // TODO: Disparar evento/email para notificar o utilizador

            } else { // Ação é 'rejeitar'
                // 1. Atualiza a transação
                $transacao->status = 'rejeitado';
                $transacao->notas_admin = $request->notas_admin;
                $transacao->save();
                
                // O saldo não é alterado, pois nunca entrou
            }
            
            DB::commit(); // Confirma a operação

            return response()->json([
                'message' => 'Depósito processado com sucesso!',
                'transacao' => $transacao
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Erro ao processar depósito.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * PROCESSA uma retirada (Aprovar ou Rejeitar)
     * (POST /api/admin/transacoes/processar-retirada/{id})
     */
  public function processarDeposito(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'acao' => 'required|in:aprovar,rejeitar',
            'notas_admin' => 'required_if:acao,rejeitar|string|nullable',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $transacao = Transacao::where('id', $id)
                            ->where('tipo_transacao', 'deposito')
                            ->where('status', 'pendente')
                            ->firstOrFail();
        
        // Puxamos o utilizador completo e a sua carteira
        $user = User::find($transacao->usuario_id);
        $userCarteira = $user->carteira;

        // --- Inicia a Transação Segura ---
        try {
            DB::beginTransaction();

            if ($request->acao === 'aprovar') {
                
                // 1. Atualiza a transação
                $transacao->status = 'concluido';
                $transacao->notas_admin = 'Depósito aprovado.';
                $transacao->save();
                
                // 2. Adiciona o saldo ao utilizador
                $userCarteira->saldo_disponivel += $transacao->valor;
                $userCarteira->save();
                
                // *** LÓGICA DE COMISSÃO DE INDICAÇÃO (A SUA SUGESTÃO) ***
                
                // 3. Verifica se este utilizador foi indicado E se este é o seu primeiro depósito
                $isFirstDeposit = $user->transacoes()
                                      ->where('tipo_transacao', 'deposito')
                                      ->where('status', 'concluido')
                                      ->count() === 1; // 1 porque esta transação já foi marcada como 'concluido'

                if ($user->indicado_por_usuario_id && $isFirstDeposit) {
                    
                    // 4. Encontra o "pai" (o indicador)
                    $indicador = User::find($user->indicado_por_usuario_id);

                    if ($indicador) {
                        // 5. Calcula a comissão
                        // (Proativo: Vamos buscar a % da tabela de Configs que criámos!)
                        $config = Configuracao::find('comissao_indicacao_percentagem');
                        $percentagem = $config ? (float) $config->valor : 5.0; // Padrão de 5%

                        $valorComissao = ($transacao->valor * $percentagem) / 100;

                        // 6. Cria o registo da comissão (como 'pendente' para o admin pagar)
                        ComissaoIndicacao::create([
                            'usuario_id' => $indicador->id, // Quem recebe a comissão
                            'indicado_usuario_id' => $user->id, // Quem gerou a comissão
                            'transacao_origem_id' => $transacao->id, // O depósito que a gerou
                            'valor_comissao' => $valorComissao,
                            'status' => 'pendente', // Entra na fila de pagamento do Admin
                        ]);
                    }
                }
                // *** FIM DA LÓGICA DE COMISSÃO ***

            } else { // Ação é 'rejeitar'
                // 1. Atualiza a transação
                $transacao->status = 'rejeitado';
                $transacao->notas_admin = $request->notas_admin;
                $transacao->save();
                // O saldo não é alterado, pois nunca entrou
            }
            
            DB::commit(); // Confirma a operação

            return response()->json([
                'message' => 'Depósito processado com sucesso!',
                'transacao' => $transacao
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Erro ao processar depósito.', 'error' => $e->getMessage()], 500);
        }
    }